/**
 * OWASP AppSensor
 * 
 * This file is part of the Open Web Application Security Project (OWASP)
 * AppSensor project. For details, please see
 * <a href="http://www.owasp.org/index.php/Category:OWASP_AppSensor_Project">
 * 	http://www.owasp.org/index.php/Category:OWASP_AppSensor_Project</a>.
 *
 * Copyright (c) 2010 - The OWASP Foundation
 * 
 * AppSensor is published by OWASP under the BSD license. You should read and accept the
 * LICENSE before you use, modify, and/or redistribute this software.
 * 
 * @author Michael Coates <a href="http://www.aspectsecurity.com">Aspect Security</a>
 * @author John Melton <a href="http://www.jtmelton.com/">jtmelton</a>
 * @created 2010
 */
namespace org.owasp.appsensor.intrusiondetection
{

    using org.owasp.appsensor;
    using org.owasp.appsensor.errors;
    using Owasp.Esapi.Errors;
    using Owasp.Esapi;
    using System;
    using System.Collections.Generic;
    using org.owasp.appsensor.intrusiondetection.reference;

    /**
     * Implementation of the IntrusionDetector interface. This
     * implementation monitors both EnterpriseSecurityExceptions as well as 
     * AppSensorExceptions to see if any user exceeds a configurable threshold in 
     * a configurable time period. For example, it can monitor to see if a user exceeds 
     * 10 input validation issues in a 1 minute period. Or if there are more than 3 
     * authentication problems in a 10 second period. 
     * <p>
     * This implementation stores state in the configured IntrusionStore class.  
     * The default implementation of the IntrusionStore (DefaultIntrusionStore) 
     * maintains the information "in-memory" so that information does not persist 
     * across application restarts.  However, more complex implementations that store 
     * the information in a file, database or other data store are certainly possible.
     * <p>
     * In order to use this class, it must be configured in the Esapi.properties file.
     * <p>
     * It is possible to override the IntrusionStore that holds the intrusion related information, 
     * or to override the ResponseAction which responds to events at the appropriate point. 
     * In order to do so, you need to modify the configuration in the appsensor.properties file.
     * <p>
     * Finally, this class uses the AppSensorSecurityConfiguration and AppSensorThreshold 
     * classes to ensure it has the appropriate information from the configuration files in order
     * to make decisions.
     * 
     * 9/14/2010 - jtm - had to add singleton config since esapi 2.0rc7 changed their 
     * management code for instantiation of singletons. This is related to bug: 
     * http://code.google.com/p/appsensor/issues/detail?id=3
     * 
     * @author Michael Coates (michael.coates .at. owasp.org) 
     *         <a href="http://www.aspectsecurity.com">Aspect Security</a>
     * @author John Melton (jtmelton .at. gmail.com)
     *         <a href="http://www.jtmelton.com/">jtmelton</a>
     * @since February 24, 2010
     * @see org.owasp.esapi.IntrusionDetector
     */
    public class AppSensorIntrusionDetector : IntrusionDetector
    {

        private static volatile IntrusionDetector singletonInstance;

        public static IntrusionDetector GetInstance()
        {
            if (singletonInstance == null)
            {
                lock (typeof(AppSensorIntrusionDetector))
                {
                    if (singletonInstance == null)
                    {
                        singletonInstance = new AppSensorIntrusionDetector();
                    }
                }
            }
            return singletonInstance;
        }

        private ASLogger logger = APPSENSOR.AsUtilities.GetLogger("AppSensorIntrusionDetector");
        private static IntrusionStore intrusionStore = null;
        private static ResponseAction responseAction = null;
        private static AppSensorSecurityConfiguration assc = (AppSensorSecurityConfiguration)AppSensorSecurityConfiguration.GetInstance();

        /**
         * Constructor that initializes the intrusion store and response action 
         */
        public AppSensorIntrusionDetector()
        {
            if (responseAction == null)
            {
                responseAction = APPSENSOR.ResponseAction;
            }

            if (intrusionStore == null)
            {
                intrusionStore = APPSENSOR.IntrusionStore;
            }
        }

        /**
         * 
         * Constructor that initializes the intrusion store and response action by 
         * taking request parameters
         *
         * @param responseAction specified response action implementation
         * @param intrusionStore specified intrusion store implementation
         */
        public AppSensorIntrusionDetector(ResponseAction responseAction, IntrusionStore intrusionStore)
        {
            AppSensorIntrusionDetector.responseAction = responseAction;
            AppSensorIntrusionDetector.intrusionStore = intrusionStore;
        }

        /**
         * {@inheritDoc}
         */
        //public override void AddEvent(String eventName)
        //{
        //    logger.Warning("Security event " + eventName + " received with message: " + logMessage);
        //    logger.Warning("Forwarding on Security event " + eventName + " as an AppSensorException");

        //    //instead of writing separate handler code, we're just going to go ahead and treat 
        //    //an event as an AppSensorException since creating one will simply cause the addExceptionMethod below
        //    //to be called, though with a bit of redirection
        //    throw new AppSensorException(eventName, logMessage, logMessage);
        //}

        /**
         * {@inheritDoc}
         */
        public override void AddException(Exception e)
        {
            //don't handle intrusion exceptions, it could get into a loop
            if (e is IntrusionException)
            {
                return;
            }

            //handle logging for specific exception types if possible, else default to just using the message
            if (e is EnterpriseSecurityException)
            {
                logger.Warning(((EnterpriseSecurityException)e).LogMessage, e);
            }
            else if (e is AppSensorException)
            {
                logger.Warning(((AppSensorException)e).LogMessage, e);
            }
            else if (e is AppSensorSystemException)
            {
                logger.Warning(((AppSensorSystemException)e).LogMessage, e);
            }
            else
            {
                logger.Warning(e.Message, e);
            }

            //add exception to storage, and get the created AppSensorIntrusion
            AppSensorIntrusion asIntrusion = intrusionStore.addExceptionToIntrusionStore(e);

            //Here we need to check if the exception we just added was a system exception
            //we should grab the intrusion record for the system user, otherwise we'll 
            //grab the intrusion record for the current user
            IntrusionRecord intrusionRecord;
            if (asIntrusion.getSecurityException() is AppSensorSystemException)
            {
                //get the system user IntrusionRecord containing all of the system user's intrusions
                intrusionRecord = intrusionStore.getIntrusionRecordForSystemUser();
            }
            else
            {
                //get the user specific IntrusionRecord containing all of this user's intrusions
                intrusionRecord = intrusionStore.getIntrusionRecordForCurrentUser();
            }

            // check if user has violated thresholds
            // 1. Check if the user has passed the limit for this type of event
            // 2. Then check if the user has passed the limit for total security events
            if (CheckUserViolation(intrusionRecord, asIntrusion.EventCode))
            {
                //went over threshold on specific event
                TakeSecurityAction(intrusionRecord, asIntrusion);
            }
            else if (CheckUserViolation(intrusionRecord, DefaultIntrusionRecord.ALL_INTRUSIONS))
            {
                //went over threshold on all events
                TakeSecurityAction(intrusionRecord, asIntrusion);
            }

        }

        /**
         * This method checks the current number of per user intrusions to see if a threshold 
         * has been crossed for either all intrusions or this specific intrusion.  If so,
         * the event is responded to.
         * @param intrusionRecord per user object holding intrusion information
         * @param eventCode code for event / detection point
         * @return if the threshold has been crossed
         */
        private bool CheckUserViolation(IntrusionRecord intrusionRecord, String eventCode)
        {
            AppSensorThreshold threshold;
            TimeSpan interval;
            try
            {
                threshold = GetAppSensorThreshold(eventCode);

                interval = threshold.MaxTimeSpan;
            }
            catch (NullReferenceException e)
            {
                // no violation configured for this specific event, still will count towards total violations
                return false;
            }

            //get number of intrusions for this eventCode
            int numIntrusions = 0;
            if (DefaultIntrusionRecord.ALL_INTRUSIONS.Equals(eventCode, StringComparison.OrdinalIgnoreCase))
            {
                numIntrusions = intrusionRecord.GetNumberOfAllIntrusions();
            }
            else
            {
                numIntrusions = intrusionRecord.GetNumberOfIntrusions(eventCode);
            }

            String logEntry = "";
            logEntry = "Checking for user violation: EventCode: " + eventCode + ", ";
            logEntry = logEntry + "Threshold: " + threshold.Event + ", Allowed Limit: " + threshold.MaxOccurences + ", ";
            logEntry = logEntry + "Total Events Recorded: " + numIntrusions;

            logger.Info(logEntry);

            //get the number of these events that have occurred in the last interval (ie. 30 seconds)
            int initialCount = intrusionRecord.GetNumberOfIntrusionsInInterval(eventCode, interval.Seconds);
            int count = initialCount;

            //4 examples for the below code
            //1. count is 5, t.count is 10 (5%10 = 5, reset count to 5 which is < 10 prints No Violation and returns false)
            //2. count is 45, t.count is 10 (45%10 = 5, reset count to 5 which is < 10 prints No Violation and returns false) 
            //3. count is 10, t.count is 10 (10%10 = 0, no reset 10 >= 10 prints Violation Observed, adds violation and returns true)
            //4. count is 30, t.count is 10 (30%10 = 0, no reset 30 >= 10 prints Violation Observed, adds violation and returns true)

            if (count % threshold.MaxOccurences != 0)
            {
                count = count % threshold.MaxOccurences;// mod count with limit - this prevents firing on event 11 when count is every 10
            }

            logger.Debug("Total Overall Events Qualifying " + initialCount);
            logger.Debug("Total Events Qualifying For Threshold " + count);

            if (count >= threshold.MaxOccurences)
            {
                logger.Info("Violation Observed for id <" + intrusionRecord.UserID + ">");
                intrusionRecord.AddViolation(eventCode);
                return true;
            }
            else
            {
                logger.Info("No Violation for id <" + intrusionRecord.UserID + ">");
                return false;
            }

        }

        /**
         * This method decides which response action to take based on those available and those 
         * already executed.  It then dispatches the ResponseAction to perform the actual response.
         * @param intrusionRecord per user object holding intrusion information
         * @param currentIntrusion object representing current intrusion.
         */
        private void TakeSecurityAction(IntrusionRecord intrusionRecord, AppSensorIntrusion currentIntrusion)
        {
            AppSensorThreshold threshold = null;
            String lastViolation = null;

            //locking to get last violation .. this will always be populated since the  
            //CheckUserViolation method is called as a guard to this method and it is the 
            //method that actually sets the last violation
            lock (this)
            {
                if (intrusionRecord.GetLastViolation() == null)
                {
                    throw new InvalidOperationException("The last Violation is null - this SHOULD NOT happen!  " +
                            "It should be set as part of determining the threshold has been crossed which occurs " +
                            "before taking security action");
                }
                lastViolation = intrusionRecord.GetLastViolation();
            }

            try
            {
                // get threshold for this event
                threshold = GetAppSensorThreshold(lastViolation);
            }
            catch (Exception e)
            {
                logger.Warning("No threshhold defined for" + lastViolation);
                // no threshold configured for this event, default to "Total" exception threshold
                threshold = GetAppSensorThreshold(DefaultIntrusionRecord.ALL_INTRUSIONS);
            }

            //get the response actions configured for this type of event 
            IList<String> responseActions = threshold.Actions;

            logger.Debug("Possible response actions for this event are: " + responseActions);

            //get the last action that has been performed if any
            //ie. if "log, logout, disable" is configured, maybe "log" has already been performed
            String lastAction = intrusionRecord.GetLastResponseAction(currentIntrusion.EventCode);

            int nextActionIndex = 0;
            if (lastAction != null)
            {
                //pull the location of the last response, and go up one to grab the next in line
                nextActionIndex = responseActions.IndexOf(lastAction) + 1;
                logger.Debug("Last executed response was: " + lastAction);
            }

            //if the last action was the most recently executed, perform it again
            if (nextActionIndex >= responseActions.Count)
            {
                nextActionIndex = responseActions.Count - 1;
                logger.Error("No further response actions defined. Repeating last action of '" + responseActions[nextActionIndex] + "'");
            }

            String action = responseActions[nextActionIndex];
            intrusionRecord.SetLastResponseAction(action, currentIntrusion.EventCode);

            responseAction.HandleResponse(action, currentIntrusion);
        }

        /**
         * Retrieves the AppSensorThreshold from the security config class for the detection point 
         * with the given eventCode 
         * @param eventCode code for event / detection point
         * @return AppSensorThreshold holding threshold information from the Esapi.properties file
         * 		   for the given eventCode
         */
        public static AppSensorThreshold GetAppSensorThreshold(String eventCode)
        {
            AppSensorThreshold threshold = assc.GetAppSensorQuota(eventCode);
            return threshold;
        }

        /**
         * Simple getter
         * @return intrusion store
         */
        public IntrusionStore GetIntrusionStore()
        {
            return intrusionStore;
        }

        /**
         * Simple getter
         * @return response action
         */
        public ResponseAction GetResponseAction()
        {
            return responseAction;
        }
    }
}